import { db } from "../db/index.js";
import { projects } from "../db/schema.js";
import { eq } from "drizzle-orm";

export const createProject = async (req, res) => {
  try {
    const { 
      title, slug, excerpt, category, stack, stage,  
      challenge, solution, howItWorks, impact, 
      description, type, liveLink, appLink
    } = req.body;
    const userId = req.user.id; // From authMiddleware

    if (!title || !slug) {
      return res.status(400).json({ message: "Title and slug are required" });
    }

    // Handle files via Multer
    let featureImage = null;
    let challengeSolutionImages = [];
    let screenshots = [];

    if (req.files) {
      if (req.files['featureImage']) {
        featureImage = `/uploads/${req.files['featureImage'][0].filename}`;
      }
      if (req.files['challengeSolutionImages']) {
        challengeSolutionImages = req.files['challengeSolutionImages'].map(file => `/uploads/${file.filename}`);
      }
      if (req.files['screenshots']) {
        screenshots = req.files['screenshots'].map(file => `/uploads/${file.filename}`);
      }
    }

    // Parse JSON fields if they come as strings from FormData
    const parsedStack = stack ? JSON.parse(stack) : null;

    const [result] = await db.insert(projects).values({
      title, slug, excerpt, category, stack: parsedStack, stage, featureImage, 
      challenge, solution, challengeSolutionImages, howItWorks, impact, 
      description, type: type || "web", liveLink, appLink, screenshots,
      userId,
    });

    res.status(201).json({ message: "Project created successfully", projectId: result.insertId });
  } catch (error) {
    console.error(error);
    if(error.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({ message: "Slug must be unique" });
    }
    res.status(500).json({ message: "Internal server error" });
  }
};

export const getProjects = async (req, res) => {
  try {
    const userId = req.user.id;
    const userProjects = await db.select().from(projects).where(eq(projects.userId, userId));
    res.status(200).json(userProjects);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

export const updateProject = async (req, res) => {
  try {
    const { id } = req.params;
    const { 
      title, slug, excerpt, category, stack, stage, 
      challenge, solution, howItWorks, impact, 
      description, type, liveLink, appLink
    } = req.body;
    const userId = req.user.id;

    // Get existing project to preserve images if not updated
    const [existingProject] = await db.select().from(projects).where(eq(projects.id, parseInt(id)));
    if (!existingProject) {
      return res.status(404).json({ message: "Project not found" });
    }

    // Handle files via Multer
    let featureImage = existingProject.featureImage;
    let challengeSolutionImages = existingProject.challengeSolutionImages || [];
    let screenshots = existingProject.screenshots || [];

    if (req.files) {
      if (req.files['featureImage']) {
        featureImage = `/uploads/${req.files['featureImage'][0].filename}`;
      }
      // Depending on your requirement, you might want to append or replace array of images.
      // Here we replace if new files are provided.
      if (req.files['challengeSolutionImages']) {
        challengeSolutionImages = req.files['challengeSolutionImages'].map(file => `/uploads/${file.filename}`);
      }
      if (req.files['screenshots']) {
        screenshots = req.files['screenshots'].map(file => `/uploads/${file.filename}`);
      }
    }

    // Parse JSON fields if they come as strings from FormData
    const parsedStack = stack ? JSON.parse(stack) : existingProject.stack;

    await db.update(projects)
      .set({ 
        title, slug, excerpt, category, stack: parsedStack, stage, featureImage, 
        challenge, solution, challengeSolutionImages, howItWorks, impact, 
        description, type, liveLink, appLink, screenshots, 
        updatedAt: new Date() 
      })
      .where(eq(projects.id, parseInt(id)))
      .where(eq(projects.userId, userId));

    res.status(200).json({ message: "Project updated successfully" });
  } catch (error) {
    console.error(error);
    if(error.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({ message: "Slug must be unique" });
    }
    res.status(500).json({ message: "Internal server error" });
  }
};

export const deleteProject = async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user.id;

    await db.delete(projects)
      .where(eq(projects.id, parseInt(id)))
      .where(eq(projects.userId, userId));

    res.status(200).json({ message: "Project deleted successfully" });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

// Public Endpoints
export const getPublicProjects = async (req, res) => {
  try {
    const allProjects = await db.select().from(projects);
    res.status(200).json(allProjects);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

export const getPublicProjectBySlug = async (req, res) => {
  try {
    const { slug } = req.params;
    const result = await db.select().from(projects).where(eq(projects.slug, slug));
    
    if (result.length === 0) {
      return res.status(404).json({ message: "Project not found" });
    }

    res.status(200).json(result[0]);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};
