import { db } from "../db/index.js";
import { services } from "../db/schema.js";
import { eq } from "drizzle-orm";

export const createService = async (req, res) => {
  try {
    const { 
      title, slug, description, subServices, 
      benefitsTitle, benefits, faqs, icon
    } = req.body;
    const userId = req.user.id;

    if (!title || !slug) {
      return res.status(400).json({ message: "Title and slug are required" });
    }

    // Handle files via Multer
    let featureImage = null;
    let benefitIcons = [];
    if (req.files) {
      if (req.files['featureImage']) {
        featureImage = `/uploads/${req.files['featureImage'][0].filename}`;
      }
      if (req.files['benefitIcons']) {
        benefitIcons = req.files['benefitIcons'].map(file => `/uploads/${file.filename}`);
      }
    }

    // Parse JSON fields
    let parsedSubServices = typeof subServices === 'string' ? JSON.parse(subServices) : subServices;
    let parsedBenefits = typeof benefits === 'string' ? JSON.parse(benefits) : benefits;
    let parsedFaqs = typeof faqs === 'string' ? JSON.parse(faqs) : faqs;

    // Attach icons to benefits if benefitIcons were uploaded
    // (We'll match them by index order: benefitIcons[0] goes to parsedBenefits[0].icon, etc.)
    if (parsedBenefits && Array.isArray(parsedBenefits) && benefitIcons.length > 0) {
      parsedBenefits = parsedBenefits.map((benefit, index) => {
        if (benefitIcons[index]) {
            return { ...benefit, icon: benefitIcons[index] };
        }
        return benefit;
      });
    }

    const [result] = await db.insert(services).values({
      title,
      slug,
      description,
      subServices: parsedSubServices,
      benefitsTitle,
      benefits: parsedBenefits,
      faqs: parsedFaqs,
      icon, // This is the main service icon/name
      featureImage,
      userId,
    });

    res.status(201).json({ message: "Service created successfully", serviceId: result.insertId });
  } catch (error) {
    console.error(error);
    if(error.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({ message: "Slug must be unique" });
    }
    res.status(500).json({ message: "Internal server error" });
  }
};

export const getServices = async (req, res) => {
  try {
    const userId = req.user.id;
    const userServices = await db.select().from(services).where(eq(services.userId, userId));
    res.status(200).json(userServices);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

export const getServiceById = async (req, res) => {
    try {
        const { id } = req.params;
        const [service] = await db.select().from(services).where(eq(services.id, parseInt(id)));
        if (!service) {
          return res.status(404).json({ message: "Service not found" });
        }
        res.status(200).json(service);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: "Internal server error" });
    }
};

export const updateService = async (req, res) => {
  try {
    const { id } = req.params;
    const { 
      title, slug, description, subServices, 
      benefitsTitle, benefits, faqs, icon
    } = req.body;
    const userId = req.user.id;

    const [existingService] = await db.select().from(services).where(eq(services.id, parseInt(id)));
    if (!existingService) {
      return res.status(404).json({ message: "Service not found" });
    }

    // Handle files via Multer
    let featureImage = existingService.featureImage;
    let benefitIcons = [];
    if (req.files) {
      if (req.files['featureImage']) {
        featureImage = `/uploads/${req.files['featureImage'][0].filename}`;
      }
      if (req.files['benefitIcons']) {
        benefitIcons = req.files['benefitIcons'].map(file => `/uploads/${file.filename}`);
      }
    }

    // Parse JSON fields
    let parsedSubServices = typeof subServices === 'string' ? JSON.parse(subServices) : subServices;
    let parsedBenefits = typeof benefits === 'string' ? JSON.parse(benefits) : (benefits || existingService.benefits);
    let parsedFaqs = typeof faqs === 'string' ? JSON.parse(faqs) : faqs;

    // Attach icons to benefits if new icons were uploaded
    if (parsedBenefits && Array.isArray(parsedBenefits) && benefitIcons.length > 0) {
      parsedBenefits = parsedBenefits.map((benefit, index) => {
        if (benefitIcons[index]) {
            return { ...benefit, icon: benefitIcons[index] };
        }
        return benefit;
      });
    }

    await db.update(services)
      .set({ 
        title, 
        slug, 
        description, 
        subServices: parsedSubServices,
        benefitsTitle, 
        benefits: parsedBenefits, 
        faqs: parsedFaqs, 
        icon,
        featureImage,
        updatedAt: new Date() 
      })
      .where(eq(services.id, parseInt(id)))
      .where(eq(services.userId, userId));

    res.status(200).json({ message: "Service updated successfully" });
  } catch (error) {
    console.error(error);
    if(error.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({ message: "Slug must be unique" });
    }
    res.status(500).json({ message: "Internal server error" });
  }
};

export const deleteService = async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user.id;

    await db.delete(services)
      .where(eq(services.id, parseInt(id)))
      .where(eq(services.userId, userId));

    res.status(200).json({ message: "Service deleted successfully" });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

// Public Endpoints
export const getPublicServices = async (req, res) => {
  try {
    const allServices = await db.select().from(services);
    res.status(200).json(allServices);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};

export const getPublicServiceBySlug = async (req, res) => {
  try {
    const { slug } = req.params;
    const result = await db.select().from(services).where(eq(services.slug, slug));
    
    if (result.length === 0) {
      return res.status(404).json({ message: "Service not found" });
    }

    res.status(200).json(result[0]);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Internal server error" });
  }
};
