# Implementation: Product Design Service CRUD & Public API (v2)

I have updated the `services` system to support **Feature Images** and **Icons for Benefits**, all handled via Multer file uploads.

## 🛠️ Updated Backend

1.  **Database schema**: Updated [schema.js](file:///d:/Level6It/level6-backend/src/db/schema.js) with `featureImage` and updated `benefits` (JSON) to store icon paths.
2.  **Controller**: Modified [serviceController.js](file:///d:/Level6It/level6-backend/src/controllers/serviceController.js) to process multiple file fields.
3.  **Routes**: Configured [serviceRoutes.js](file:///d:/Level6It/level6-backend/src/routes/serviceRoutes.js) to use Multer's `upload.fields`.

---

## 🚀 API Documentation

### Public Endpoints
-   **GET** `/api/services/public` - Fetch all services.
-   **GET** `/api/services/public/:slug` - Fetch specific service by slug.

### Admin Endpoints (Auth Token Required)
-   **POST** `/api/services` - Create (Supports `featureImage` and `benefitIcons`).
-   **PUT** `/api/services/:id` - Update.
-   **DELETE** `/api/services/:id` - Delete.

---

## 🎨 Frontend Integration (Multipart/Form-Data)

To send icons and feature images, you **must** use `FormData`.

### Payload (Form Data)
-   `title`: "Product Design"
-   `slug`: "product-design"
-   `description`: "User-centered design..."
-   `icon`: "main-icon-class" (Optional text)
-   **`featureImage`**: (File) - Main image for the service.
-   **`benefitIcons`**: (Files) - Array of icon files for the benefits.
-   `benefits`: (Stringified JSON) - `[{"title": "...", "description": "..."}, ...]`
    -   *Note: The icons will be automatically mapped to these benefits based on their order in `benefitIcons`.*
-   `faqs`: (Stringified JSON) - `[{"question": "...", "answer": "..."}, ...]`

### React Example

```javascript
const formData = new FormData();
formData.append('title', 'Product Design');
formData.append('slug', 'product-design');
formData.append('description', 'User-centered design...');
formData.append('icon', 'target-icon'); // Main service icon (String)

// Store the main feature image
formData.append('featureImage', featureFile);

// Store icons for each benefit (handled by index)
formData.append('benefitIcons', icon1File);
formData.append('benefitIcons', icon2File);

// JSON data (Order of benefitIcons must match benefits order)
formData.append('benefits', JSON.stringify([
    {
      "title": "Fast, not flimsy",
      "description": "Launch a stable MVP...",
      "icon": "" // Will be populated with the uploaded benefitIcons[0] path
    },
    {
      "title": "Senior execution",
      "description": "Product led by people...",
      "icon": "" // Will be populated with the uploaded benefitIcons[1] path
    }
]));

formData.append('faqs', JSON.stringify([
    { question: "What's included?", answer: "Detailed services list." }
]));

const res = await fetch('/api/services', {
    method: 'POST',
    headers: {
        'Authorization': `Bearer ${token}`
    },
    body: formData
});
```

> [!TIP]
> All uploaded images will be stored in `/uploads/` and served via `http://localhost:5000/uploads/filename`.
